<?php
/***************************************************************************
*                                                                          *
*    Copyright (c) 2009 Simbirsk Technologies Ltd. All rights reserved.    *
*                                                                          *
* This  is  commercial  software,  only  users  who have purchased a valid *
* license  and  accept  to the terms of the  License Agreement can install *
* and use this program.                                                    *
*                                                                          *
****************************************************************************
* PLEASE READ THE FULL TEXT  OF THE SOFTWARE  LICENSE   AGREEMENT  IN  THE *
* "copyright.txt" FILE PROVIDED WITH THIS DISTRIBUTION PACKAGE.            *
****************************************************************************/

//
// $Id: func.php by tommy from cs-cart.jp 2016
//
// *** 関数名の命名ルール ***
// 混乱を避けるため、フックポイントで動作する関数とその他の命名ルールを明確化する。
// (1) init.phpで定義ししたフックポイントで動作する関数：fn_localization_jp_[フックポイント名]
// (2) addons.xmlで定義した設定項目で動作する関数：fn_settings_variants_addons_localization_jp_[アイテムID]
// (3) (1)以外の関数：fn_lcjp_[任意の名称]

// Modified by takahashi from cs-cart.jp 2018
// $paymentをチェックする関数を追加
// 商品検索の設定を変更
// SEO作成時にピリオドがサニタイズされない問題を修正

// Modified by takahashi from cs-cart.jp 2019
// 軽減税率対応（あんどぷらす望月様よりご提供）

use Tygh\Registry;
use Tygh\Helpdesk;

if (!defined('BOOTSTRAP')) { die('Access denied'); }

//////////////////////////////////////////////////////////
// PDF納品書に関する関数 START
//////////////////////////////////////////////////////////

// 顧客情報の追加フィールド項目を取得
function fn_my_changes_get_add_profile_fields($section, $type)
{
    // 指定無し
    $variants = array('0' => __('jp_pdfinv_not_assigned'));

    // ユーザー追加フィールドのID、名称を取得
    $extra_fields = db_get_array("SELECT ?:profile_fields.field_id, ?:profile_field_descriptions.description FROM ?:profile_fields LEFT JOIN ?:profile_field_descriptions ON ?:profile_fields.field_id = ?:profile_field_descriptions.object_id WHERE ?:profile_fields.class like '%_kana' AND ?:profile_fields.section = ?s  AND ?:profile_fields.field_type = ?s AND ?:profile_field_descriptions.lang_code = ?s AND ?:profile_field_descriptions.object_type = ?s", $section, $type, DESCR_SL, 'F');

    if ($extra_fields) {
        foreach($extra_fields as $fields){
            $variants[$fields['field_id']] = $fields['description'];
        }
    }
    return $variants;
}




// PDF納品書出力
function fn_my_changes_print_pdf_invoice($order_ids)
{
    //////////////////////////////////////////////
    // PDF出力ライブラリの設定 BOF
    //////////////////////////////////////////////
    require(Registry::get('config.dir.addons') . 'localization_jp/lib/tcpdf/config/lang/jpn.php');
    require(Registry::get('config.dir.addons') . 'localization_jp/lib/tcpdf/tcpdf.php');
    $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
    $pdf->setPrintHeader(false);
    $pdf->setPrintFooter(false);
    $pdf->SetMargins(5, 7, 5);
    $pdf->SetAutoPageBreak(false);
    $pdf->setJPEGQuality(75);
    $base_line_width = 0.201083333333;
    //////////////////////////////////////////////
    // PDF出力ライブラリの設定 EOF
    //////////////////////////////////////////////

    // 1注文分のみの出力の場合、引数に含まれる注文IDを配列にセット
    if( !is_array($order_ids) ){
        $temp_order_id = $order_ids;
        $order_ids = array();
        $order_ids[] = $temp_order_id;
    }

    // 注文バーコードの設定情報を取得
    $barcode_settings = Registry::get('addons.barcode');

    // ショップ情報
    $out_shop_data = array();

    // ショップ名
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_name')) != '' ){
        $out_shop_data[] = Registry::get('addons.localization_jp.jp_pdfinv_shop_name');
    }
    // ショップ郵便番号
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_zipcode')) != '' ) {
        $out_shop_data[] = __('jp_pdfinv_zip_title') . ' ' . Registry::get('addons.localization_jp.jp_pdfinv_shop_zipcode');
    }
    // ショップ住所上段
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_address_up')) != '' ) {
        $out_shop_data[] = Registry::get('addons.localization_jp.jp_pdfinv_shop_address_up');
    }
    // ショップ住所下段
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_address_dw')) != '' ) {
        $out_shop_data[] = Registry::get('addons.localization_jp.jp_pdfinv_shop_address_dw');
    }
    // ショップ担当者
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_person_in_charge')) != '' ) {
        $out_shop_data[] = __('jp_pdfinv_person_in_charge') . __('jp_pdfinv_full_colon') . Registry::get('addons.localization_jp.jp_pdfinv_shop_person_in_charge');
    }
    // e-mail
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_contact_mail')) != '' ) {
        $out_shop_data[] = __('email') . __('jp_pdfinv_full_colon') . Registry::get('addons.localization_jp.jp_pdfinv_shop_contact_mail');
    }
    // 電話番号/FAX番号
    $buf = '';
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_contact_phone')) != '' ) {
        $buf = __('phone') . __('jp_pdfinv_full_colon') . Registry::get('addons.localization_jp.jp_pdfinv_shop_contact_phone');
    }
    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_contact_fax')) != '' ) {
        if( $buf != '' ) $buf .= ' / ';
        $buf .= __('fax') . __('jp_pdfinv_full_colon') . Registry::get('addons.localization_jp.jp_pdfinv_shop_contact_fax');
    }
    if( $buf != '' ) $out_shop_data[] = $buf;

    // ショップ通信欄（下段）行数
    $info_line_count = 0;
    $info_line_count = fn_my_changes_adjust_customer_comment($pdf, Registry::get('addons.localization_jp.jp_pdfinv_print_info_dw'), $dummy);

    // 出力ファイル名用変数を初期化
    $output_order_id = '';

    // 出力ファイル用注文ID
    foreach( $order_ids as $order_id ){

        // 注文情報取得
        $order_data = fn_get_order_info($order_id);

        $output_order_id .= '_' . $order_id;

        // ロゴ画像のパスを取得
        $logos = fn_get_logos($order_data['company_id']);
        $filename = $logos['mail']['image']['absolute_path'];

        if(!is_file($filename)) {
            $_cid = Registry::get('runtime.forced_company_id');
            if(!empty($_cid)) {
                $logos = fn_get_logos($_cid);
            } else {
                $logos = fn_get_logos(0);
            }
            $filename = $logos['mail']['image']['absolute_path'];
        }

        // 注文時の通貨取得
        $order_currency_setting = db_get_row("SELECT data FROM ?:order_data WHERE order_id = ?i AND type = 'R'", $order_id);
        if( !empty($order_currency_setting) ){
            $order_currency = unserialize($order_currency_setting['data']);
        } else {
            $order_currency = CART_PRIMARY_CURRENCY;
        }

        // 注文合計ページのみ印刷が必要か
        $order_total_page_print = false;

        // 明細行を減らす必要があるか
        $product_line_minus_flag = false;
        $product_line_minus_count = 0;

        // ページ数計算
        $total_pages = floor(count($order_data['products']) / 10);

        // 最終ページの明細行
        $fragment_line = (count($order_data['products']) % 10);
        $total_pages += ($fragment_line > 0 ? 1 : 0);

        // 注文合計行の計算
        $order_total_line = fn_my_changes_count_total_line($order_data);

        if( $order_total_line > 8 ){
            // 合計行が8行以上ある場合は商品明細行を少なくする
            $over_count = $order_total_line - 8;
            $over_line = floor($over_count / 3);
            $over_line += (($over_count % 3) > 0 ? 1 : 0);
            if( $fragment_line == 0 || ($fragment_line + $over_line) > 10 ){
                $order_total_page_print = true;
                $total_pages++;
            }else{
                $product_line_minus_flag = true;
                $product_line_minus_count = $over_line;
            }
        }

        $print_line = 99;
        // ページ小計
        $page_total = 0;
        $pages = 0;

        //////////////////////////////////////////////
        // 明細行印刷 BOF
        //////////////////////////////////////////////
        foreach($order_data['products'] as $products_array_key => $product) {
            // 改ページ処理
            if( $print_line > 9 ){
                // ページを跨がる印刷に場合はページ小計を印刷
                if( $page_total > 0 ){
                    $pdf->SetX(120);
                    $pdf->MultiCell(40, 0, __('jp_pdfinv_page_subtotal'), 1, 'R', 1, 0);
                    $print_price = fn_my_changes_convert_price($page_total, $order_currency, true);
                    $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
                }
                if( $pages > 0 ){
                    // ページ印刷
                    $pdf->SetXY(200, 288);
                    $pdf->Cell(0, 0, __('pages') . ' ' . $pages . '/' . $total_pages, 0, 0, 'R');
                    if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_url')) != '' ){
                        $pdf->SetXY(5, 288);
                        $shop_url = __('website') . ' ' . trim(mb_convert_kana(Registry::get('addons.localization_jp.jp_pdfinv_shop_url'), "as", "UTF-8"));
                        $font_size = fn_my_changes_get_url_font_size($pdf, 8, $shop_url);
                        $pdf->SetFont('kozminproregular', '', $font_size);
                        $pdf->Cell(0, 0, $shop_url, 0, 0, 'L');
                        $pdf->SetFont('kozminproregular', '', 8);
                    }
                    // バーコード印刷
                    if( !empty($barcode_settings) && $barcode_settings['status'] != 'D' ){
                        $bcStyle = array(
                            'position' => 'L',
                            'border' => false,
                            'padding' => 0,
                            'fgcolor' => array(0,0,0),
                            'bgcolor' => false,
                            'text' => true,
                            'font' => 'kozminproregular',
                            'fontsize' => 6,
                            'stretchtext' => 4
                        );
                        if( $barcode_settings['type'] != 'C128C' && $barcode_settings['type'] != 'I25' ){
                            $output_code = $barcode_settings['prefix'] . $order_data["order_id"];
                        } else {
                            $output_code = $order_data["order_id"];
                        }
                        $pdf->write1DBarcode($output_code, $barcode_settings['type'], 5, 5, 50, 8, 0.4, $bcStyle, 'N');
                    }
                }
                // ヘッダー印刷
                fn_my_changes_print_pdf_header($pdf, $out_shop_data, $filename, $barcode_settings, $order_data);
                $print_line = 0;
                // ページ小計
                $page_total = 0;
                $pages++;
            }

            // カスタマイズ商品対応（親商品のみメインループで印刷）
            if( !$product['extra']['parent'] || !empty($product['extra']['parent']['buy_together'])){
                // 商品単価
                $print_base_price = $product['original_price'];
                // 注文小計
                $print_subtotal = $product['display_subtotal'];
                $page_total += $print_subtotal;
                // 子商品があるか確認し、あれば価格を合算
                $custome_product = 0;
                foreach($order_data['products'] as $subitem){
                    if( !empty($subitem['extra']['parent']['configuration']) && $subitem['extra']['parent']['configuration'] == $products_array_key ){
                        $custome_product = 1;
                        $print_base_price += $subitem['original_price'];
                        $print_subtotal += $subitem['display_subtotal'];
                    }
                }
                fn_my_changes_print_order_line($pdf, $product, $print_base_price, $print_subtotal, $order_currency, $custome_product);
                $print_line++;
                if( $custome_product == 1 ){
                    foreach($order_data['items'] as $subitem) {
                        if( $print_line > 9 ){
                            // ヘッダー印刷
                            fn_my_changes_print_pdf_header($pdf, $out_shop_data, $filename, $barcode_settings, $order_data);
                            $print_line = 0;
                        }
                        if( !empty($subitem['extra']['parent']['configuration']) && $subitem['extra']['parent']['configuration'] == $products_array_key ){
                            $custome_product = 2;
                            // 商品単価
                            $print_base_price = $subitem['original_price'];
                            // 注文小計
                            $print_subtotal = $subitem['display_subtotal'];
                            fn_my_changes_print_order_line($pdf, $subitem, $print_base_price, $print_subtotal, $order_currency, $custome_product);
                            $print_line++;
                        }
                    }
                }
            }
        }

        if( $order_total_page_print ){
            // ページを跨がる印刷に場合はページ小計を印刷
            if( $page_total > 0 ){
                $pdf->SetX(120);
                $pdf->MultiCell(40, 0, __('jp_pdfinv_page_subtotal'), 1, 'R', 1, 0);
                $print_price = fn_my_changes_convert_price($page_total, $order_currency, true);
                $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
            }
            if( $pages > 0 ){
                // ページ印刷
                $pdf->SetXY(200, 288);
                $pdf->Cell(0, 0, __('pages') . ' ' . $pages . '/' . $total_pages, 0, 0, 'R');
                if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_url')) != '' ){
                    $pdf->SetXY(5, 288);
                    $shop_url = __('website') . ' ' . trim(mb_convert_kana(Registry::get('addons.localization_jp.jp_pdfinv_shop_url'), "as", "UTF-8"));
                    $font_size = fn_my_changes_get_url_font_size($pdf, 8, $shop_url);
                    $pdf->SetFont('kozminproregular', '', $font_size);
                    $pdf->Cell(0, 0, $shop_url, 0, 0, 'L');
                    $pdf->SetFont('kozminproregular', '', 8);
                }
                // バーコード印刷
                if( !empty($barcode_settings) && $barcode_settings['status'] != 'D' ){
                    $bcStyle = array(
                        'position' => 'L',
                        'border' => false,
                        'padding' => 0,
                        'fgcolor' => array(0,0,0),
                        'bgcolor' => false,
                        'text' => true,
                        'font' => 'kozminproregular',
                        'fontsize' => 6,
                        'stretchtext' => 4
                    );
                    if( $barcode_settings['type'] != 'C128C' && $barcode_settings['type'] != 'I25' ){
                        $output_code = $barcode_settings['prefix'] . $order_data["order_id"];
                    } else {
                        $output_code = $order_data["order_id"];
                    }
                    $pdf->write1DBarcode($output_code, $barcode_settings['type'], 5, 5, 50, 8, 0.4, $bcStyle, 'N');
                }
            }
            // ヘッダー印刷
            fn_my_changes_print_pdf_header($pdf, $out_shop_data, $filename, $barcode_settings, $order_data);
            $pages++;
        }else{
            if( $product_line_minus_flag ){
                $page_max_line = 10 - $product_line_minus_count;
            } else {
                $page_max_line = 10;
            }
            if( $print_line < $page_max_line ){
                for($lp=$print_line; $lp<$page_max_line; $lp++) {
                    $pdf->MultiCell(115, 16.5840833334, '', 1, 'L', 0, 0);
                    $pdf->MultiCell(25, 16.5840833334, '', 1, 'L', 0, 0);
                    $pdf->MultiCell(15, 16.5840833334, '', 1, 'L', 0, 0);
                    $pdf->MultiCell(20, 16.5840833334, '', 1, 'L', 0, 0);
                    $pdf->MultiCell(25, 16.5840833334, '', 1, 'L', 0, 1);
                }
            }
        }
        //////////////////////////////////////////////
        // 明細行印刷 EOF
        //////////////////////////////////////////////

        //////////////////////////////////////////////
        // 備考欄印刷 BOF
        //////////////////////////////////////////////
        $_memo_info = '';

        // 支払方法を備考欄に記載
        if ( !empty($order_data['payment_method']['payment']) ){
            $_memo_info = '【' . __('payment_method') . '】 ' . $order_data['payment_method']['payment'];
        }

        // 配送に関する情報を備考欄に記載
        $lcjp_shipping_info = '';
        if( !empty($order_data['shipping']) && is_array($order_data['shipping']) ){
            foreach($order_data['shipping'] as $shipping_data){
                // 配送方法名が登録されている場合
                if( !empty($shipping_data['shipping']) ) $lcjp_shipping_info .= "\n" . '【' . __('shipping_method') . '】 ' . $shipping_data['shipping'];
                if( !empty($shipping_data['delivery_date']) )  $lcjp_shipping_info .= "\n" . __('jp_delivery_date') . ' : ' . $shipping_data['delivery_date'];
                if( !empty($shipping_data['delivery_timing']) )  $lcjp_shipping_info .= "\n" . __('jp_shipping_delivery_time') . ' : ' . $shipping_data['delivery_timing'] . "\n";
            }
        }
        if( !empty($lcjp_shipping_info) ){
            $_memo_info .= "\n" . $lcjp_shipping_info;
        }

        // お客様コメントを備考欄に記載
        if( !empty($order_data['notes']) ){
            // お客様が入力したコメントから改行を削除
            $_customer_comments = str_replace("\r\n", '\n', $order_data['notes']);
            $_customer_comments = str_replace("\r", '\n', $_customer_comments);
            $_customer_comments = str_replace('\n', '', $_customer_comments);
            $_memo_info .= "\n" . '【' . __('customer_notes') . '】' . "\n" . $_customer_comments;
        }

        $outpuf_note = $_memo_info . "\n\n" . Registry::get('addons.localization_jp.jp_pdfinv_print_info_dw');
        $pdf->MultiCell(115, 44.224222222, $outpuf_note, 1, 'L', 0, 0, '', '', true, 0, false, true, 46.224222222);
        //////////////////////////////////////////////
        // 備考欄印刷 EOF
        //////////////////////////////////////////////

        //////////////////////////////////////////////
        // 合計欄印刷 BOF
        //////////////////////////////////////////////
        // 小計
        $pdf->SetX(120);
        $pdf->MultiCell(40, 0, __('subtotal'), 1, 'R', 1, 0);
        $print_price = fn_my_changes_convert_price($order_data['display_subtotal'], $order_currency, true);
        $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);

        // 値引き
        if( $order_data['discount'] > 0 ){
            $pdf->SetX(120);
            $pdf->MultiCell(40, 0, __('including_discount'), 1, 'R', 1, 0);
            $print_price = fn_my_changes_convert_price($order_data['discount'], $order_currency, true);
            $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
        }

        // 注文値引き
        if( $order_data['subtotal_discount'] > 0 ){
            $pdf->SetX(120);
            $pdf->MultiCell(40, 0, __('order_discount'), 1, 'R', 1, 0);
            $print_price = fn_my_changes_convert_price($order_data['subtotal_discount'], $order_currency, true);
            $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
        }

        // クーポン印刷
        if( $order_data['coupons'] ){
            $pdf->SetFont('kozminproregular', '', 7);
            foreach($order_data['coupons'] as $key => $val) {
                $pdf->SetX(120);
                $pdf->MultiCell(40, 0, __('coupons'), 1, 'R', 1, 0);
                $pdf->MultiCell(45, 0, $key, 1, 'R', 0, 1);
            }
            $pdf->SetFont('kozminproregular', '', 8);
        }

        // 送料
        $shipping_total = 0;
        if( !empty($order_data['shipping']) ){
            foreach($order_data['shipping'] as $shipping) {
                $shipping_total += $shipping['rate'];
            }
        }
        if( $shipping_total > 0 ){
            $pdf->SetX(120);
            $pdf->MultiCell(40, 0, __('shipping_cost'), 1, 'R', 1, 0);
            $print_price = fn_my_changes_convert_price($shipping_total, $order_currency, true);
            $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
        }

        // 代引き手数料
        if( !empty($order_data['payment_surcharge']) && $order_data['payment_surcharge'] > 0 ){
            $pdf->SetX(120);
            $pdf->MultiCell(40, 0, __('payment_surcharge'), 1, 'R', 1, 0);
            $print_price = fn_my_changes_convert_price($order_data['payment_surcharge'], $order_currency, true);
            $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
        }

        // ポイント情報
        if( (int)$order_data['points_info']['reward'] > 0 ){
            $pdf->SetX(120);
            $pdf->MultiCell(40, 0, __('points'), 1, 'R', 1, 0);
            $pdf->MultiCell(45, 0, number_format((int)$order_data['points_info']['reward']) . __('reward_points'), 1, 'R', 0, 1);
        }
        if( (int)$order_data['points_info']['in_use']['cost'] > 0 ){
            $pdf->SetFont('kozminproregular', '', 7);
            $pdf->SetX(120);
            $pdf->MultiCell(40, 0, __('points_in_use') . '(' . number_format((int)$order_data['points_info']['in_use']['points']) . __('points') . ')', 1, 'R', 1, 0);
            $print_price = fn_my_changes_convert_price($order_data['points_info']['in_use']['cost'], $order_currency, true);
            $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
            $pdf->SetFont('kozminproregular', '', 8);
        }
        // ギフト券
        if( $order_data['use_gift_certificates'] ){
            $pdf->SetFont('kozminproregular', '', 7);
            foreach($order_data['use_gift_certificates'] as $key => $val){
                $pdf->SetX(120);
                $pdf->MultiCell(40, 0, __('gift_certificate') . '[' . $key . ']', 1, 'R', 1, 0);
                $print_price = fn_my_changes_convert_price($val['cost'], $order_currency, true);
                $pdf->MultiCell(45, 0,'(' . $print_price . ')', 1, 'R', 0, 1);
            }
            $pdf->SetFont('kozminproregular', '', 8);
        }

        ///////////////////////////////////////////////////////////////
        // Modified for Japanese Ver by takahashi from cs-cart.jp 2019 BOF
        // 軽減税率対応（あんどぷらす望月様よりご提供）
        ///////////////////////////////////////////////////////////////
        foreach ($order_data['taxes'] as $tax) {
            if ((int) $tax['rate_value'] > 0) {
                $pdf->SetX(120);
                $pdf->MultiCell(40, 0, $tax['description'] . ' ' . number_format($tax['rate_value']) . '%', 1, 'R', 1, 0);
                $print_price = fn_my_changes_convert_price($tax['tax_subtotal'], $order_currency, true);
                $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
            }
        }
        ///////////////////////////////////////////////////////////////
        // Modified for Japanese Ver by takahashi from cs-cart.jp 2019 EOF
        ///////////////////////////////////////////////////////////////

        // 合計
        $pdf->SetFont('kozminproregular', 'B', 9);
        $pdf->SetX(120);
        $pdf->MultiCell(40, 0, __('total_cost'), 1, 'R', 1, 0);
        $print_price = fn_my_changes_convert_price($order_data['total'], $order_currency, true);
        $pdf->MultiCell(45, 0, $print_price, 1, 'R', 0, 1);
        $pdf->SetFont('kozminproregular', '', 8);
        //////////////////////////////////////////////
        // 合計欄印刷 EOF
        //////////////////////////////////////////////

        // ページ印刷
        $pdf->SetXY(200, 288);
        $pdf->Cell(0, 0, __('pages') . ' ' . $pages . '/' . $total_pages, 0, 0, 'R');
        if( trim(Registry::get('addons.localization_jp.jp_pdfinv_shop_url')) != '' ){
            $pdf->SetXY(5, 288);
            $shop_url = __('website') . ' ' . trim(mb_convert_kana(Registry::get('addons.localization_jp.jp_pdfinv_shop_url'), "as", "UTF-8"));
            $font_size = fn_my_changes_get_url_font_size($pdf, 8, $shop_url);
            $pdf->SetFont('kozminproregular', '', $font_size);
            $pdf->Cell(0, 0, $shop_url, 0, 0, 'L');
            $pdf->SetFont('kozminproregular', '', 8);
        }

        // バーコード
        if( !empty($barcode_settings) && $barcode_settings['status'] != 'D' ){
            $bcStyle = array(
                'position' => 'L',
                'border' => false,
                'padding' => 0,
                'fgcolor' => array(0,0,0),
                'bgcolor' => false,
                'text' => true,
                'font' => 'kozminproregular',
                'fontsize' => 6,
                'stretchtext' => 4
            );
            if( $barcode_settings['type'] != 'C128C' && $barcode_settings['type'] != 'I25' ){
                $output_code = $barcode_settings['prefix'] . $order_data["order_id"];
            }else{
                $output_code = $order_data["order_id"];
            }
            $pdf->write1DBarcode($output_code, $barcode_settings['type'], 5, 5, 50, 8, 0.4, $bcStyle, 'N');
        }
    }

    // 出力ファイル名をセット
    $output_filename = "order_invoice" . $output_order_id . ".pdf";

    // PDFファイルを出力
    $pdf->Output($output_filename, "D");
}




// 明細行印刷
function fn_my_changes_print_order_line(&$pdf, $product, $print_base_price, $print_subtotal, $order_currency, $custome_product = NULL)
{
    // 商品名（カスタム商品の子であれば商品名、オプション名の先頭に半角スペース2個追加）
    $product_data = ($custome_product == 2 ? '  ' : '') . fn_my_changes_adjust_strings($pdf, $product['product']);
    // 商品オプション
    $add_line = 0;
    if( $custome_product == 1 ){
        $max_add_line = 1;
    }else{
        $max_add_line = 2;
    }
    $option_price = 0;
    $over_count = 0;
    $over_count = (count($product['product_options']) - ($max_add_line + 1));
    if( count($product['product_options']) > 0 ){
        foreach($product['product_options'] as $option){
            if( $add_line <= $max_add_line ){
                $product_data .= "\n" . fn_my_changes_adjust_strings($pdf, '  ' . $option['option_name'] . __('jp_pdfinv_full_colon') . $option['variant_name'], 'option');
            }
            $option_price += ($option['modifier'] > 0 ? $option['modifier'] : 0);
            $add_line++;
        }
    }else{
        // 定期支払い
        // オプションがない場合のみ印刷
        if( $product['extra']['recurring_plan_id'] && (int)$custome_product == 0 ){
            // プラン名
            $product_data .= "\n" . __('rb_recurring_plan') . __('jp_pdfinv_full_colon') . $product['extra']['recurring_plan']['name'];
            // 支払い間隔
            $product_data .= "\n" . __('rb_recurring_period') . __('jp_pdfinv_full_colon') . fn_get_recurring_period_name($product['extra']['recurring_plan']['period']);
            // 支払い期間
            $product_data .= "\n" . __('rb_start_duration') . __('jp_pdfinv_full_colon') . $product['extra']['recurring_duration'];
        }
    }
    // その他表示
    if( $over_count > 0 ){
        $product_data .= ' ' . __('jp_pdfinv_other_option');
    }
    if( $custome_product == 1 ){
        $product_data .= "\n" . __('jp_pdfinv_customise_product');
    }
    $pdf->MultiCell(115, 16.5840833334, $product_data, 1, 'L', 0, 0);
    // 商品型番
    $pdf->MultiCell(25, 16.5840833334, $product['product_code'], 1, 'L', 0, 0);
    // 数量
    $pdf->MultiCell(15, 16.5840833334, number_format($product['amount']), 1, 'R', 0, 0);
    // 単価
    if( $product['extra']['exclude_from_calculate'] ){
        $pdf->MultiCell(20, 16.5840833334, __('free'), 1, 'R', 0, 0);
    }else{
        $print_price = fn_my_changes_convert_price(($print_base_price), $order_currency, false);
        // 多通貨対応 円、＄以外の通貨はコードを出力するためフォントサイズを調整
        $_font_sieze = 8;
        while($pdf->GetStringWidth($print_price, 'kozminproregular', '', $_font_sieze) > 18.2) {
            $_font_sieze = $_font_sieze - 0.2;
        }
        if($option_price != 0) {
            $print_price .= "\n(" . fn_my_changes_convert_price(($option_price), $order_currency, false) . ')';
        }
        $pdf->SetFont('kozminproregular', '', $_font_sieze);
        $pdf->MultiCell(20, 16.5840833334, $print_price, 1, 'R', 0, 0, '', '', true, 1);
        $pdf->SetFont('kozminproregular', '', 8);
    }

    // 税抜きの小計を表示
    if( $product['extra']['exclude_from_calculate'] ){
        $pdf->MultiCell(25, 16.5840833334, __('free'), 1, 'R', 0, 1);
    }else{
        $print_item_price = fn_my_changes_convert_price($print_subtotal, $order_currency, false);
        // 値引きがあれば印刷
        $discount_info = '';
        if( (int)$product['discount'] > 0 ){
            $discount_total = ($product['discount'] * $product['amount']);
            $_discount_price = __('discount') . ' ' . fn_my_changes_convert_price($discount_total, $order_currency, false);
            // 多通貨対応 円、＄以外の通貨はコードを出力するためフォントサイズを調整
            while($pdf->GetStringWidth($_discount_price, 'kozminproregular', '', $_font_sieze) > 23.2) {
                $_font_sieze = $_font_sieze - 0.2;
            }
            $pdf->SetFont('kozminproregular', '', $_font_sieze);
            $print_item_discount_price = "\n" . $_discount_price;
        }
        $print_price = $print_item_price . $print_item_discount_price;
        $pdf->MultiCell(25, 16.5840833334, $print_price, 1, 'R', 0, 1);
        $pdf->SetFont('kozminproregular', '', 8);
    }
}




// 商品明細行の文字列調整
function fn_my_changes_adjust_strings(&$pdf, $strings, $type = NULL)
{
    if( $type == NULL ){
        $max_width = 110;
    }elseif( $type == 'option' ){
        $max_width = 79;
    }elseif( $type == 'comment' ){
        $max_width = 196;
    }
    $work_string = str_replace("\r\n", "\n", $strings);
    $work_string = str_replace("\r", "\n", $work_string);
    $work_string = str_replace("\n", '', $work_string);
    if( $pdf->GetStringWidth($work_string, 'kozminproregular', '', 8) < $max_width ){
        return $work_string;
    }else{
        $lp = 0;
        $res = '';
        $buf = '';
        $buf = mb_substr($work_string, 0, 1, 'UTF-8');
        while($pdf->GetStringWidth($res . $buf, 'kozminproregular', '', 8) < $max_width){
            $res .= $buf;
            $lp++;
            if( $lp > mb_strlen($work_string, 'UTF-8') ) break;
            $buf = mb_substr($work_string, $lp, 1, 'UTF-8');
        }
        return $res;
    }
}




// 顧客追加情報の取得
function fn_my_changes_get_customer_extra_field_data($order_id, $field_id)
{
    return db_get_field("SELECT ?:profile_fields_data.value FROM ?:profile_fields_data WHERE ?:profile_fields_data.object_type = 'O' AND ?:profile_fields_data.object_id = ?i AND ?:profile_fields_data.field_id = ?i", $order_id, $field_id);
}




// 合計行のライン数計算
function fn_my_changes_count_total_line($order_data)
{
    // 小計、合計行で2行
    $ret_count = 2;
    // 値引き
    if( (int)$order_data['discount'] > 0 ) $ret_count++;

    // 注文値引き
    if( (int)$order_data['subtotal_discount'] > 0 ) $ret_count++;

    // クーポン印刷
    if( $order_data['coupons'] ){
        foreach($order_data['coupons'] as $key => $val){
            $ret_count++;
        }
    }

    // 送料
    /* !!!!!!!! 後で検証する !!!!!!!!!!!!!!!!
    $shipping_total = 0;

    if( !empty($order_data['shipping']) ){
        foreach($order_data['shipping'] as $shipping){
            foreach($shipping['rates'] as $shipping_rates){
                $shipping_total += $shipping_rates;
            }
        }
    }

    if( (int)$shipping_total > 0 ) $ret_count++;
    */

    // 代引き手数料
    if( !empty($order_data['payment_surcharge']) && $order_data['payment_surcharge'] > 0 ) $ret_count++;

    // 取得ポイント情報
    if( (int)$order_data['points_info']['reward'] > 0 ) $ret_count++;
    // 利用ポイント
    if( (int)$order_data['points_info']['in_use']['cost'] > 0 ) $ret_count++;
    // ギフト券
    if( $order_data['use_gift_certificates'] ){
        foreach($order_data['use_gift_certificates'] as $key => $val){
            $ret_count++;
        }
    }
    // 消費税
    $tax_total = 0;
    foreach($order_data['taxes'] as $tax){
        if( (int)$tax['rate_value'] > 0 ){
            $tax_total += $tax['tax_subtotal'];
        }
    }
    if( $tax_total > 0 ) $ret_count++;

    return $ret_count;
}




// コメント欄の整形
function fn_my_changes_adjust_customer_comment($pdf, $customer_comment, &$res_comment)
{
    $res_comment = array();
    $cr_cnv_buf = str_replace("\r\n", "\n", trim($customer_comment));
    $cr_cnv_buf = str_replace("\r", "\n", $cr_cnv_buf);
    $cr_cnv_buf = str_replace("\n", '<>', $cr_cnv_buf);
    if( $cr_cnv_buf != '' ){
        $work_comment = explode('<>', $cr_cnv_buf);
        $line_count = 0;
        foreach($work_comment as $work_line){
            if( trim($work_line) != '' ){
                $line_count++;
                $res_comment[($line_count - 1)] = '';
                for($lp=0; $lp<mb_strlen(trim($work_line), 'UTF-8'); $lp++){
                    $res_comment[($line_count - 1)] .= mb_substr(trim($work_line), $lp, 1, 'UTF-8');
                    if( $pdf->GetStringWidth($res_comment[($line_count - 1)], 'kozminproregular', '', 8) > 110.066666667 ){
                        $line_count++;
                        $res_comment[($line_count - 1)] = '';
                    }
                }
            }
        }
        return $line_count;
    }else{
        return 0;
    }
}




// pdf納品書ヘッダー出力
function fn_my_changes_print_pdf_header(&$pdf, $out_shop_data, $filename, $barcode_settings, $order_data)
{
    $pdf->AddPage();

    $pdf->SetFont('kozminproregular', 'B', 24);
    $pdf->Cell(0, 0, __('jp_pdfinv_invoice'), 0, 1, 'C');
    $pdf->SetFont('kozminproregular', '', 8);
    $pdf->SetLineStyle(array('width' => 2, 'cap' => 'square', 'color' => array(0, 0, 0)));
    $pdf->Line(5, 18, 205, 18);
    $pdf->SetLineStyle(array('width' => 0.201083333333, 'cap' => 'square', 'color' => array(0, 0, 0)));
    $order_id_width = $pdf->GetStringWidth(__('order_id') . __('jp_pdfinv_full_colon') . $order_data['order_id'], 'kozminproregular', '', 8);
    $pdf->SetXY(203.5 - $order_id_width, 8);
    $pdf->Cell(0, 0, __('order_id') . __('jp_pdfinv_full_colon') . $order_data['order_id'], 1, 1, 'R');
    $pdf->SetXY(185, 12.55);
    $pdf->Cell(0, 0, __('order_date') . __('jp_pdfinv_full_colon') . date("Y/m/d", $order_data['timestamp']), 0, 1, 'R');
    $pdf->SetY(20);

    $image_size = getimagesize($filename);
    $logo_width = ($image_size[0] / 2.834);
    $logo_height = ($image_size[1] / 2.834);

    // 顧客情報
    $out_customer = array();

    // 支払いと配送情報（氏名、住所）が違う場合
    if( $order_data['b_firstname'] != $order_data['s_firstname'] || $order_data['b_lastname'] != $order_data['s_lastname'] || $order_data['b_address'] != $order_data['s_address'] ){
        if( Registry::get('addons.localization_jp.jp_pdfinv_print_customer_address') == 0 ){
            // 請求先住所を印刷
            // 郵便番号
            $out_customer[] = __('jp_pdfinv_zip_title') . ' ' . $order_data['b_zipcode'];
            // 住所上段
            $out_customer[] = $order_data['b_state'] . $order_data['b_city'];
            // 住所下段
            $out_customer[] = $order_data['b_address'] . $order_data['b_address_2'];

            // 顧客会社名
            $out_company_name = '';
            if( (int)Registry::get('addons.localization_jp.jp_pdfinv_billing_company_field') > 0 ){
                $out_company_name = fn_my_changes_get_customer_extra_field_data($order_data['order_id'], Registry::get('addons.localization_jp.jp_pdfinv_billing_company_field'));
            }
            if( $out_company_name != '' ){
                $out_customer[] = $out_company_name;
            }else{
                if( $order_data['company'] != '' ){
                    $out_customer[] = $order_data['company'];
                }
            }

            // 顧客名
            $out_customer[] = $order_data['b_firstname'] . ' ' . $order_data['b_lastname'] . ' ' . __('dear');

            // 顧客電話番号
            $out_customer_phone = $order_data['b_phone'];

            if( !empty($out_customer_phone) ){
                $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $out_customer_phone;

            }elseif( (int)Registry::get('addons.localization_jp.jp_pdfinv_billing_phone_field') > 0 ){
                $out_customer_phone = fn_my_changes_get_customer_extra_field_data($order_data['order_id'], Registry::get('addons.localization_jp.jp_pdfinv_billing_phone_field'));

                if( !empty($out_customer_phone) ){
                    $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $out_customer_phone;
                }else{
                    if( !empty($order_data['phone']) ){
                        $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $order_data['phone'];
                    }
                }
            }else{
                if( !empty($order_data['phone']) ){
                    $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $order_data['phone'];
                }
            }

        }else{
            // 配送先住所を印刷
            // 郵便番号
            $out_customer[] = __('jp_pdfinv_zip_title') . ' ' . $order_data['s_zipcode'];
            // 住所上段
            $out_customer[] = $order_data['s_state'] . $order_data['s_city'];
            // 住所下段
            $out_customer[] = $order_data['s_address'] . $order_data['s_address_2'];
            // 顧客会社名
            $out_company_name = '';
            if( (int)Registry::get('addons.localization_jp.jp_pdfinv_shipping_company_field') > 0 ){
                $out_company_name = fn_my_changes_get_customer_extra_field_data($order_data['order_id'], Registry::get('addons.localization_jp.jp_pdfinv_shipping_company_field'));
            }
            if( $out_company_name != '' ){
                $out_customer[] = $out_company_name;
            }
            // 顧客名
            $out_customer[] = $order_data['s_firstname'] . ' ' . $order_data['s_lastname'] . ' ' . __('dear');

            // 顧客電話番号
            $out_customer_phone = $order_data['s_phone'];

            if( !empty($out_customer_phone) ){
                $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $out_customer_phone;

            }elseif( (int)Registry::get('addons.localization_jp.jp_pdfinv_shipping_phone_field') > 0 ){
                $out_customer_phone = fn_my_changes_get_customer_extra_field_data($order_data['order_id'], Registry::get('addons.localization_jp.jp_pdfinv_shipping_phone_field'));

                if( !empty($out_customer_phone) ){
                    $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $out_customer_phone;
                }else{
                    if( !empty($order_data['phone']) ){
                        $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $order_data['phone'];
                    }
                }
            }else{
                if( !empty($order_data['phone']) ){
                    $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $order_data['phone'];
                }
            }
        }
    }else{
        // 請求先住所を印刷
        // 郵便番号
        $out_customer[] = __('jp_pdfinv_zip_title') . ' ' . $order_data['b_zipcode'];
        // 住所上段
        $out_customer[] = $order_data['b_state'] . $order_data['b_city'];
        // 住所下段
        $out_customer[] = $order_data['b_address'] . $order_data['b_address_2'];

        // 顧客会社名
        $out_company_name = '';
        if( (int)Registry::get('addons.localization_jp.jp_pdfinv_billing_company_field') > 0 ){
            $out_company_name = fn_my_changes_get_customer_extra_field_data($order_data['order_id'], Registry::get('addons.localization_jp.jp_pdfinv_billing_company_field'));
        }
        if( $out_company_name != '' ){
            $out_customer[] = $out_company_name;
        }else{
            if( $order_data['company'] != '' ){
                $out_customer[] = $order_data['company'];
            }
        }
        // 顧客名
        $out_customer[] = $order_data['b_firstname'] . ' ' . $order_data['b_lastname'] . ' ' . __('dear');

        // 顧客電話番号
        $out_customer_phone = $order_data['b_phone'];

        if( !empty($out_customer_phone) ){
            $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $out_customer_phone;

        }elseif( (int)Registry::get('addons.localization_jp.jp_pdfinv_billing_phone_field') > 0 ){
            $out_customer_phone = fn_my_changes_get_customer_extra_field_data($order_data['order_id'], Registry::get('addons.localization_jp.jp_pdfinv_billing_phone_field'));

            if( !empty($out_customer_phone) ){
                $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $out_customer_phone;
            }else{
                if( !empty($order_data['phone']) ){
                    $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $order_data['phone'];
                }
            }
        }else{
            if( !empty($order_data['phone']) ){
                $out_customer[] = __('phone') . __('jp_pdfinv_full_colon') . $order_data['phone'];
            }
        }
    }
    $customer_font_size = 10;
    $shop_font_size = 10;
    for($lp=0; $lp<7; $lp++){
        if( isset($out_customer[$lp]) && !empty($out_customer[$lp]) ){
            if( $pdf->GetStringWidth($out_customer[$lp], 'kozminproregular', '', 10) > 100 ){
                $customer_font_size = 8;
            }
        }
        if( isset($out_shop_data[$lp]) && !empty($out_shop_data[$lp]) ){
            if( $pdf->GetStringWidth($out_shop_data[$lp], 'kozminproregular', '', 10) > 100 ){
                $shop_font_size = 8;
            }
        }
    }
    $pdf->SetFont('kozminproregular', '', 10);
    // 顧客、ショップ情報印刷
    for($lp=0; $lp<7; $lp++){
        if( isset($out_customer[$lp]) && !empty($out_customer[$lp]) ){
            $pdf->SetFont('kozminproregular', '', $customer_font_size);
            $pdf->MultiCell(100, 0, $out_customer[$lp], 0, 'L', 0, 0);
        }else{
            // インデント
            $pdf->SetX(105);
        }
        $pdf->SetFont('kozminproregular', '', 10);
        if( $lp==0 ){
            // ロゴ印刷
            if( Registry::get('addons.localization_jp.jp_pdfinv_print_shop_logo') == 'Y' ){
                $pdf->Image($filename, (205 - $logo_width), 20, 0, 0, '', '', '', false, 600);
            }
        }
        if( isset($out_shop_data[$lp]) && !empty($out_shop_data[$lp]) ){
            $pdf->SetFont('kozminproregular', '', $shop_font_size);
            $pdf->MultiCell(100, 0, $out_shop_data[$lp], 0, 'L', 0, 1);
        }else{
            // 改行
            $pdf->MultiCell(100, 0, '', 0, 'L', 0, 1);
        }
        $pdf->SetFont('kozminproregular', '', 10);
    }

    // ショップ側コメント上段印刷（改行はさせない）
    $pdf->SetXY(5, 64.8698055556);
    $shop_comment = str_replace("\r\n", "\n", trim(Registry::get('addons.localization_jp.jp_pdfinv_print_info_up')));
    $shop_comment = str_replace("\r", "\n", $shop_comment);
    $shop_comment = str_replace("\n", '', $shop_comment);
    $shop_comment = fn_my_changes_adjust_strings($pdf, trim($shop_comment), 'comment');
    $pdf->MultiCell(0, 0, $shop_comment, 0, 'L', 0, 1);		// ショップ担当者、電話番号/FAX番号

    $pdf->SetXY(5, 71.2797777778);
    $pdf->SetFont('kozminproregular', '', 8);
    // 明細行見出し
    $pdf->SetFillColor(221, 254, 255);
    $pdf->SetLineStyle(array('width' => $base_line_width, 'cap' => 'square', 'color' => array(11, 249, 255)));
    $pdf->MultiCell(115, 0, __('products') . ' / ' . __('options'), 1, 'C', 1, 0);
    $pdf->MultiCell(25, 0, __('sku'), 1, 'C', 1, 0);
    $pdf->MultiCell(15, 0, __('quantity'), 1, 'C', 1, 0);
    $pdf->MultiCell(20, 0, __('unit_price'), 1, 'C', 1, 0);
    $pdf->MultiCell(25, 0, __('subtotal'), 1, 'C', 1, 1);
}




// 通貨情報に従って表示金額文字列を作成
function fn_my_changes_convert_price($price, $print_currency, $total_line = false)
{
    if( Registry::get('settings.General.alternative_currency') == 'Y' ){
        $res = fn_my_changes_format_price_by_currency($price, CART_PRIMARY_CURRENCY);
        if( $print_currency != CART_PRIMARY_CURRENCY ){
            $res .= (!$total_line ? "\n" : '') . '(' . fn_my_changes_format_price_by_currency($price, $print_currency) . ')';
        }
    }else{
        $res = fn_my_changes_format_price_by_currency($price, $print_currency);
    }
    return $res;
}




// 表示通貨にて円とドル以外は通貨コードを利用する
function fn_my_changes_format_price_by_currency($price, $currency_code = CART_SECONDARY_CURRENCY)
{
    $currencies = Registry::get('currencies');
    $currency = $currencies[$currency_code];
    $result = fn_format_rate_value($price, 'F', $currency['decimals'], $currency['decimals_separator'], $currency['thousands_separator'], $currency['coefficient']);
    if ($currency['after'] == 'Y') {
        if(strtoupper($currency['currency_code']) == 'JPY' || strtoupper($currency['currency_code']) == 'USD') {
            $result .= ' ' . $currency['symbol'];
        } else {
            $result .= ' ' . $currency['currency_code'];
        }
    } else {
        if(strtoupper($currency['currency_code']) == 'JPY' || strtoupper($currency['currency_code']) == 'USD') {
            $result = $currency['symbol'] . $result;
        } else {
            $result = $currency['currency_code'] . $result;
        }
    }

    return $result;
}




// ショップURLのフォントサイズ取得
function fn_my_changes_get_url_font_size($pdf, $default_size, $url_string)
{
    $work_size = $default_size;
    while($pdf->GetStringWidth($url_string, 'kozminproregular', '', $work_size) > 180){
        $work_size = $work_size - 0.5;
        if( $work_size <= 0 ){
            $work_size = 1;
            break;
        }
    }
    return $work_size;
}

//////////////////////////////////////////////////////////
// PDF納品書に関する関数 END
//////////////////////////////////////////////////////////